/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.plugins.index.importer;

import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import org.apache.commons.collections4.ListValuedMap;
import org.apache.commons.collections4.multimap.ArrayListValuedHashMap;
import org.apache.jackrabbit.oak.api.CommitFailedException;
import org.apache.jackrabbit.oak.api.PropertyState;
import org.apache.jackrabbit.oak.api.Type;
import org.apache.jackrabbit.oak.commons.conditions.Validate;
import org.apache.jackrabbit.oak.commons.time.Stopwatch;
import org.apache.jackrabbit.oak.plugins.index.FormattingUtils;
import org.apache.jackrabbit.oak.plugins.index.IndexEditorProvider;
import org.apache.jackrabbit.oak.plugins.index.IndexUpdate;
import org.apache.jackrabbit.oak.plugins.index.IndexUpdateCallback;
import org.apache.jackrabbit.oak.plugins.index.IndexUtils;
import org.apache.jackrabbit.oak.plugins.index.IndexingReporter;
import org.apache.jackrabbit.oak.plugins.index.MetricsFormatter;
import org.apache.jackrabbit.oak.plugins.index.MetricsUtils;
import org.apache.jackrabbit.oak.plugins.index.importer.AsyncIndexerLock;
import org.apache.jackrabbit.oak.plugins.index.importer.AsyncLaneSwitcher;
import org.apache.jackrabbit.oak.plugins.index.importer.IndexDefinitionUpdater;
import org.apache.jackrabbit.oak.plugins.index.importer.IndexImporterProvider;
import org.apache.jackrabbit.oak.plugins.index.importer.IndexerInfo;
import org.apache.jackrabbit.oak.plugins.index.importer.NodeStoreUtils;
import org.apache.jackrabbit.oak.plugins.index.upgrade.IndexDisabler;
import org.apache.jackrabbit.oak.plugins.memory.PropertyStates;
import org.apache.jackrabbit.oak.spi.commit.EditorDiff;
import org.apache.jackrabbit.oak.spi.commit.VisibleEditor;
import org.apache.jackrabbit.oak.spi.state.NodeBuilder;
import org.apache.jackrabbit.oak.spi.state.NodeState;
import org.apache.jackrabbit.oak.spi.state.NodeStateUtils;
import org.apache.jackrabbit.oak.spi.state.NodeStore;
import org.apache.jackrabbit.oak.stats.StatisticsProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class IndexImporter {
    static final String ASYNC_LANE_SYNC = "sync";
    static final String TYPE_ELASTICSEARCH = "elasticsearch";
    public static final String OAK_INDEX_IMPORTER_PRESERVE_CHECKPOINT = "oak.index.importer.preserveCheckpoint";
    private static final Logger LOG = LoggerFactory.getLogger(IndexImporter.class);
    private final NodeStore nodeStore;
    private final File indexDir;
    private final Map<String, IndexImporterProvider> importers = new HashMap<String, IndexImporterProvider>();
    private final IndexerInfo indexerInfo;
    private final Map<String, File> indexes;
    private final ListValuedMap<String, IndexInfo> asyncLaneToIndexMapping;
    private final NodeState indexedState;
    private final IndexEditorProvider indexEditorProvider;
    private final AsyncIndexerLock indexerLock;
    private final IndexDefinitionUpdater indexDefinitionUpdater;
    private final boolean preserveCheckpoint = Boolean.getBoolean("oak.index.importer.preserveCheckpoint");
    static final int RETRIES = Integer.getInteger("oak.index.import.retries", 5);
    public static final String INDEX_IMPORT_STATE_KEY = "indexImportState";
    private final Set<String> indexPathsToUpdate;
    private final StatisticsProvider statisticsProvider;
    private final IndexingReporter indexingReporter;

    public IndexImporter(NodeStore nodeStore, File indexDir, IndexEditorProvider indexEditorProvider, AsyncIndexerLock indexerLock) throws IOException {
        this(nodeStore, indexDir, indexEditorProvider, indexerLock, StatisticsProvider.NOOP, IndexingReporter.NOOP);
    }

    public IndexImporter(NodeStore nodeStore, File indexDir, IndexEditorProvider indexEditorProvider, AsyncIndexerLock indexerLock, StatisticsProvider statisticsProvider) throws IOException {
        this(nodeStore, indexDir, indexEditorProvider, indexerLock, statisticsProvider, IndexingReporter.NOOP);
    }

    public IndexImporter(NodeStore nodeStore, File indexDir, IndexEditorProvider indexEditorProvider, AsyncIndexerLock indexerLock, StatisticsProvider statisticsProvider, IndexingReporter indexingReporter) throws IOException {
        this.statisticsProvider = statisticsProvider;
        this.indexingReporter = indexingReporter;
        Validate.checkArgument(indexDir.exists() && indexDir.isDirectory(), "Path [%s] does not point to existing directory", indexDir.getAbsolutePath());
        this.nodeStore = nodeStore;
        this.indexDir = indexDir;
        this.indexEditorProvider = indexEditorProvider;
        this.indexerInfo = IndexerInfo.fromDirectory(indexDir);
        this.indexerLock = indexerLock;
        this.indexes = this.indexerInfo.getIndexes();
        this.indexedState = Objects.requireNonNull(nodeStore.retrieve(this.indexerInfo.checkpoint), "Cannot retrieve checkpointed state [" + this.indexerInfo.checkpoint + "]");
        this.indexDefinitionUpdater = new IndexDefinitionUpdater(new File(indexDir, "index-definitions.json"));
        this.asyncLaneToIndexMapping = this.mapIndexesToLanes(this.indexes);
        this.indexPathsToUpdate = new HashSet<String>();
    }

    public void importIndex() throws IOException, CommitFailedException {
        try {
            this.indexingReporter.setIndexNames(List.copyOf(this.indexes.keySet()));
            if (this.indexes.isEmpty()) {
                LOG.warn("No indexes to import (possibly index definitions outside of a oak:index node?)");
            }
            LOG.info("Proceeding to import {} indexes from {}", this.indexes.keySet(), (Object)this.indexDir.getAbsolutePath());
            this.runWithRetry(RETRIES, IndexImportState.SWITCH_LANE, this::switchLanes);
            LOG.info("Done with switching of index lanes before import");
            this.runWithRetry(RETRIES, IndexImportState.IMPORT_INDEX_DATA, this::importIndexData);
            LOG.info("Done with importing of index data");
            this.runWithRetry(RETRIES, IndexImportState.BRING_INDEX_UPTODATE, this::bringIndexUpToDate);
            LOG.info("Done with bringing index up-to-date");
            this.runWithRetry(RETRIES, IndexImportState.RELEASE_CHECKPOINT, this::releaseCheckpoint);
            LOG.info("Done with releasing checkpoint");
            this.updateIndexImporterState(IndexImportState.RELEASE_CHECKPOINT, null, true);
            LOG.info("Done with removing index import state");
        }
        catch (IOException | CommitFailedException e) {
            LOG.error("Failure while index import", (Throwable)e);
            try {
                this.runWithRetry(RETRIES, null, () -> {
                    NodeState root = this.nodeStore.getRoot();
                    NodeBuilder builder = root.builder();
                    this.revertLaneChange(builder, this.indexPathsToUpdate);
                    NodeStoreUtils.mergeWithConcurrentCheck(this.nodeStore, builder);
                });
            }
            catch (CommitFailedException commitFailedException) {
                LOG.error("Unable to revert back index lanes for: {}", (Object)this.indexPathsToUpdate.stream().collect(StringBuilder::new, StringBuilder::append, (a, b) -> a.append(",").append((CharSequence)b)), (Object)commitFailedException);
                throw e;
            }
        }
    }

    public void addImporterProvider(IndexImporterProvider importerProvider) {
        this.importers.put(importerProvider.getType(), importerProvider);
    }

    void switchLanes() throws CommitFailedException {
        try {
            NodeState root = this.nodeStore.getRoot();
            NodeBuilder builder = root.builder();
            for (IndexInfo indexInfo : this.asyncLaneToIndexMapping.values()) {
                if (indexInfo.newIndex) continue;
                NodeBuilder idxBuilder = NodeStoreUtils.childBuilder(builder, indexInfo.indexPath);
                this.indexPathsToUpdate.add(indexInfo.indexPath);
                String idxBuilderType = idxBuilder.getString("type");
                if (idxBuilderType != null && !idxBuilderType.equals(indexInfo.type) && (idxBuilderType.equals(TYPE_ELASTICSEARCH) || indexInfo.type.equals(TYPE_ELASTICSEARCH))) {
                    LOG.info("Provided index [{}] has a different type compared to the existing index. Using lane from the index definition provided", (Object)indexInfo.indexPath);
                    PropertyState asyncProperty = PropertyStates.createProperty("async", List.of(indexInfo.asyncLaneName), Type.STRINGS);
                    idxBuilder.setProperty(asyncProperty);
                }
                AsyncLaneSwitcher.switchLane(idxBuilder, AsyncLaneSwitcher.getTempLaneName(indexInfo.asyncLaneName));
            }
            this.updateIndexImporterState(builder, IndexImportState.NULL, IndexImportState.SWITCH_LANE, false);
            NodeStoreUtils.mergeWithConcurrentCheck(this.nodeStore, builder);
        }
        catch (CommitFailedException e) {
            LOG.error("Failed while performing switchLanes and updating indexImportState from  [{}] to  [{}]", (Object)IndexImportState.NULL, (Object)IndexImportState.SWITCH_LANE);
            throw e;
        }
    }

    void importIndexData() throws CommitFailedException, IOException {
        try {
            NodeState root = this.nodeStore.getRoot();
            NodeBuilder rootBuilder = root.builder();
            IndexDisabler indexDisabler = new IndexDisabler(rootBuilder);
            for (IndexInfo indexInfo : this.asyncLaneToIndexMapping.values()) {
                LOG.info("Importing index data for {}", (Object)indexInfo.indexPath);
                NodeBuilder idxBuilder = this.indexDefinitionUpdater.apply(rootBuilder, indexInfo.indexPath);
                if (indexInfo.newIndex) {
                    AsyncLaneSwitcher.switchLane(idxBuilder, AsyncLaneSwitcher.getTempLaneName(indexInfo.asyncLaneName));
                    this.indexPathsToUpdate.add(indexInfo.indexPath);
                } else {
                    NodeState existing = NodeStateUtils.getNode(root, indexInfo.indexPath);
                    IndexImporter.copyLaneProps(existing, idxBuilder);
                }
                this.incrementReIndexCount(idxBuilder);
                this.getImporter(indexInfo.type).importIndex(root, idxBuilder, indexInfo.indexDir);
                indexDisabler.markDisableFlagIfRequired(indexInfo.indexPath, idxBuilder);
            }
            this.updateIndexImporterState(root.builder(), IndexImportState.SWITCH_LANE, IndexImportState.IMPORT_INDEX_DATA, false);
            NodeStoreUtils.mergeWithConcurrentCheck(this.nodeStore, rootBuilder, this.indexEditorProvider);
        }
        catch (CommitFailedException e) {
            LOG.error("Failed while performing importIndexData and updating indexImportState from  [{}] to  [{}]", (Object)IndexImportState.SWITCH_LANE, (Object)IndexImportState.IMPORT_INDEX_DATA);
            throw e;
        }
    }

    private void bringIndexUpToDate() throws CommitFailedException {
        for (String laneName : this.asyncLaneToIndexMapping.keySet()) {
            if (ASYNC_LANE_SYNC.equals(laneName)) continue;
            this.bringAsyncIndexUpToDate(laneName, this.asyncLaneToIndexMapping.get((Object)laneName));
        }
    }

    private void bringAsyncIndexUpToDate(String laneName, List<IndexInfo> indexInfos) throws CommitFailedException {
        AsyncIndexerLock.LockToken lockToken = this.interruptCurrentIndexing(laneName);
        boolean success = false;
        try {
            String checkpoint = this.getAsync().getString(laneName);
            Objects.requireNonNull(checkpoint, "No current checkpoint found for lane [" + laneName + "]");
            NodeState after = this.nodeStore.retrieve(checkpoint);
            Objects.requireNonNull(after, "No state found for checkpoint [" + checkpoint + "] for lane [" + laneName + "]");
            LOG.info("Proceeding to update imported indexes {} to checkpoint [{}] for lane [{}]", new Object[]{indexInfos, checkpoint, laneName});
            NodeState before = this.indexedState;
            NodeBuilder builder = this.nodeStore.getRoot().builder();
            IndexUpdate indexUpdate = new IndexUpdate(this.indexEditorProvider, AsyncLaneSwitcher.getTempLaneName(laneName), this.nodeStore.getRoot(), builder, IndexUpdateCallback.NOOP);
            CommitFailedException exception = EditorDiff.process(VisibleEditor.wrap(indexUpdate), before, after);
            if (exception != null) {
                throw exception;
            }
            this.revertLaneChange(builder, indexInfos);
            this.updateIndexImporterState(builder, IndexImportState.IMPORT_INDEX_DATA, IndexImportState.BRING_INDEX_UPTODATE, false);
            NodeStoreUtils.mergeWithConcurrentCheck(this.nodeStore, builder);
            success = true;
            LOG.info("Imported index is updated to repository state at checkpoint [{}] for indexing lane [{}]", (Object)checkpoint, (Object)laneName);
        }
        catch (CommitFailedException e) {
            LOG.error("Failed while performing bringIndexUpToDate and updating indexImportState from  [{}] to  [{}]", (Object)IndexImportState.IMPORT_INDEX_DATA, (Object)IndexImportState.BRING_INDEX_UPTODATE);
            throw e;
        }
        finally {
            block11: {
                try {
                    this.resumeCurrentIndexing(lockToken);
                }
                catch (RuntimeException | CommitFailedException e) {
                    LOG.warn("Error occurred while releasing indexer lock", (Throwable)e);
                    if (!success) break block11;
                    throw e;
                }
            }
        }
        LOG.info("Import done for indexes {}", indexInfos);
    }

    private void revertLaneChange(NodeBuilder builder, List<IndexInfo> indexInfos) {
        for (IndexInfo info : indexInfos) {
            NodeBuilder idxBuilder = NodeStoreUtils.childBuilder(builder, info.indexPath);
            AsyncLaneSwitcher.revertSwitch(idxBuilder, info.indexPath);
        }
    }

    private void revertLaneChange(NodeBuilder builder, Set<String> indexPaths) {
        for (String indexPath : indexPaths) {
            NodeBuilder idxBuilder = NodeStoreUtils.childBuilder(builder, indexPath);
            AsyncLaneSwitcher.revertSwitch(idxBuilder, indexPath);
        }
    }

    private IndexImportState getIndexImportState(NodeBuilder nodeBuilder) {
        if (nodeBuilder.getProperty(INDEX_IMPORT_STATE_KEY) == null || nodeBuilder.getProperty(INDEX_IMPORT_STATE_KEY).getValue(Type.STRING) == null) {
            return IndexImportState.NULL;
        }
        return IndexImportState.valueOf(nodeBuilder.getProperty(INDEX_IMPORT_STATE_KEY).getValue(Type.STRING));
    }

    private void updateIndexImporterState(IndexImportState currentImportState, IndexImportState nextImportState, boolean shouldCommit) throws CommitFailedException {
        NodeState root = this.nodeStore.getRoot();
        NodeBuilder builder = root.builder();
        this.updateIndexImporterState(builder, currentImportState, nextImportState, shouldCommit);
    }

    private void updateIndexImporterState(NodeBuilder builder, IndexImportState currentImportState, IndexImportState nextImportState, boolean shouldCommit) throws CommitFailedException {
        for (String indexPath : this.indexPathsToUpdate) {
            NodeBuilder idxBuilder = NodeStoreUtils.childBuilder(builder, indexPath);
            IndexImportState nodeStoreIndexImportState = this.getIndexImportState(idxBuilder);
            if (nodeStoreIndexImportState != currentImportState) continue;
            if (nextImportState == IndexImportState.NULL) {
                idxBuilder.removeProperty(INDEX_IMPORT_STATE_KEY);
                continue;
            }
            idxBuilder.setProperty(INDEX_IMPORT_STATE_KEY, nextImportState.toString(), Type.STRING);
        }
        if (shouldCommit) {
            NodeStoreUtils.mergeWithConcurrentCheck(this.nodeStore, builder);
        }
    }

    private void resumeCurrentIndexing(AsyncIndexerLock.LockToken lockToken) throws CommitFailedException {
        this.indexerLock.unlock(lockToken);
    }

    private AsyncIndexerLock.LockToken interruptCurrentIndexing(String laneName) throws CommitFailedException {
        return this.indexerLock.lock(laneName);
    }

    private IndexImporterProvider getImporter(String type) {
        IndexImporterProvider provider = this.importers.get(type);
        return Objects.requireNonNull(provider, "No IndexImporterProvider found for type [" + type + "]");
    }

    private ListValuedMap<String, IndexInfo> mapIndexesToLanes(Map<String, File> indexes) {
        NodeState rootState = this.nodeStore.getRoot();
        ArrayListValuedHashMap map = new ArrayListValuedHashMap();
        for (Map.Entry<String, File> e : indexes.entrySet()) {
            String indexPath = e.getKey();
            NodeState indexState = this.indexDefinitionUpdater.getIndexState(indexPath);
            Validate.checkArgument(indexState.exists(), "No index node found at path [%s]", indexPath);
            boolean newIndex = !NodeStateUtils.getNode(rootState, indexPath).exists();
            String type = indexState.getString("type");
            Objects.requireNonNull(type, "No 'type' property found for index at path [" + indexPath + "]");
            String asyncName = IndexImporter.getAsyncLaneName(indexPath, indexState);
            if (asyncName == null) {
                asyncName = ASYNC_LANE_SYNC;
            }
            map.put((Object)asyncName, (Object)new IndexInfo(indexPath, e.getValue(), asyncName, type, newIndex));
        }
        return map;
    }

    private static void copyLaneProps(NodeState existing, NodeBuilder indexBuilder) {
        IndexImporter.copy("async", existing, indexBuilder);
        IndexImporter.copy("async-previous", existing, indexBuilder);
    }

    private static void copy(String propName, NodeState existing, NodeBuilder indexBuilder) {
        PropertyState ps = existing.getProperty(propName);
        if (ps != null) {
            indexBuilder.setProperty(ps);
        }
    }

    static String getAsyncLaneName(String indexPath, NodeState indexState) {
        PropertyState asyncPrevious = indexState.getProperty("async-previous");
        if (asyncPrevious != null && !AsyncLaneSwitcher.isNone(asyncPrevious)) {
            return IndexUtils.getAsyncLaneName(indexState, indexPath, asyncPrevious);
        }
        return IndexUtils.getAsyncLaneName(indexState, indexPath);
    }

    private void releaseCheckpoint() throws CommitFailedException {
        if (this.preserveCheckpoint) {
            LOG.info("Preserving the referred checkpoint [{}]. This could have been done in case this checkpoint is needed by a process later on. Please make sure to remove the checkpoint once it's no longer needed.", (Object)this.indexerInfo.checkpoint);
            this.updateIndexImporterState(IndexImportState.BRING_INDEX_UPTODATE, null, true);
        } else if (this.nodeStore.release(this.indexerInfo.checkpoint)) {
            LOG.info("Released the referred checkpoint [{}]", (Object)this.indexerInfo.checkpoint);
            this.updateIndexImporterState(IndexImportState.BRING_INDEX_UPTODATE, IndexImportState.RELEASE_CHECKPOINT, true);
        }
    }

    private void incrementReIndexCount(NodeBuilder definition) {
        long count = 0L;
        PropertyState reindexCountProp = definition.getProperty("reindexCount");
        if (reindexCountProp != null) {
            count = reindexCountProp.getValue(Type.LONG);
        }
        definition.setProperty("reindexCount", count + 1L);
    }

    private NodeState getAsync() {
        return this.nodeStore.getRoot().getChildNode(":async");
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    void runWithRetry(int maxRetries, IndexImportState indexImportState, IndexImporterStepExecutor step) throws CommitFailedException, IOException {
        String indexImportPhaseName = indexImportState == null ? "null" : indexImportState.toString();
        int count = 1;
        Stopwatch start = Stopwatch.createStarted();
        IndexUtils.INDEXING_PHASE_LOGGER.info("[TASK:{}:START]", (Object)indexImportPhaseName);
        try {
            while (count <= maxRetries) {
                LOG.info("IndexImporterStepExecutor:{}, count:{}", (Object)indexImportPhaseName, (Object)count);
                try {
                    step.execute();
                    long durationSeconds = start.elapsed(TimeUnit.SECONDS);
                    IndexUtils.INDEXING_PHASE_LOGGER.info("[TASK:{}:END] Metrics: {}", (Object)indexImportPhaseName, (Object)MetricsFormatter.createMetricsWithDurationOnly(durationSeconds));
                    MetricsUtils.setCounterOnce(this.statisticsProvider, "oak_indexer_import_" + indexImportPhaseName.toLowerCase() + "_duration_seconds", durationSeconds);
                    this.indexingReporter.addTiming("oak_indexer_import_" + indexImportPhaseName.toLowerCase(), FormattingUtils.formatToSeconds(durationSeconds));
                    this.indexingReporter.addMetric("oak_indexer_import_" + indexImportPhaseName.toLowerCase() + "_duration_seconds", durationSeconds);
                    return;
                }
                catch (IOException | CommitFailedException e) {
                    LOG.warn("IndexImporterStepExecutor: {} fail count: {}, retries left: {}", new Object[]{indexImportState, count, maxRetries - count, e});
                    if (count++ < maxRetries) continue;
                    LOG.warn("IndexImporterStepExecutor: {} failed after {} retries", new Object[]{indexImportState, maxRetries, e});
                    throw e;
                    return;
                }
            }
        }
        catch (Throwable t) {
            IndexUtils.INDEXING_PHASE_LOGGER.info("[TASK:{}:FAIL] Metrics: {}, Error: {}", new Object[]{indexImportPhaseName, MetricsFormatter.createMetricsWithDurationOnly(start), t.toString()});
            throw t;
        }
    }

    static interface IndexImporterStepExecutor {
        public void execute() throws CommitFailedException, IOException;
    }

    private static class IndexInfo {
        final String indexPath;
        final File indexDir;
        final String asyncLaneName;
        final String type;
        final boolean newIndex;

        private IndexInfo(String indexPath, File indexDir, String asyncLaneName, String type, boolean newIndex) {
            this.indexPath = indexPath;
            this.indexDir = indexDir;
            this.asyncLaneName = asyncLaneName;
            this.type = type;
            this.newIndex = newIndex;
        }

        public String toString() {
            return this.indexPath;
        }
    }

    static enum IndexImportState {
        NULL,
        SWITCH_LANE,
        IMPORT_INDEX_DATA,
        BRING_INDEX_UPTODATE,
        RELEASE_CHECKPOINT;

    }
}

